/*
 *  Currency.h
 *  AgileBook
 *
 *  Created by James Coplien on 7/31/09.
 *  Copyright 2009 Gertrud & Cope. All rights reserved.
 *
 */


#ifndef _CURRENCY_H
#define _CURRENCY_H

#include <string>
#include <sstream>

class Currency;

class CurrencyBase {
friend class Currency;
public:
	virtual CurrencyBase &operator+=(const Currency& amount) = 0;
	virtual CurrencyBase &operator-=(const Currency& amount) = 0;
	virtual CurrencyBase &operator=(const Currency& amount) = 0;
	CurrencyBase(void): referenceCount_(1) { }
	virtual ~CurrencyBase() {  }
	virtual std::string name(void) const = 0;
	virtual std::string sign(void) const = 0;
	virtual double amountInEuro(void) const = 0;
	virtual std::string asString(void) const = 0;
protected:
	unsigned referenceCount_;
};


class Currency {
public:
	virtual Currency &operator+=(const Currency& amount) {
		*actualCurrency_ += amount;
		return *this;
	}
	virtual Currency &operator-=(const Currency& amount) {
		*actualCurrency_ -= amount;
		return *this;
	}
	virtual Currency &operator=(const Currency& amount) {
		amount.actualCurrency_->referenceCount_++;
		if (--actualCurrency_->referenceCount_ <= 0) delete actualCurrency_;
		actualCurrency_ = amount.actualCurrency_;
		return *this;
	}
	Currency(const Currency& amount) {
		(actualCurrency_ = amount.actualCurrency_)->referenceCount_++;
	}
	Currency(void);
	virtual ~Currency() { if (--actualCurrency_->referenceCount_ <= 0) delete actualCurrency_; }
	virtual std::string name(void) const { return actualCurrency_->name(); }
	virtual std::string sign(void) const { return actualCurrency_->sign(); }
	virtual double amountInEuro(void) const { return actualCurrency_->amountInEuro(); }
	virtual std::string asString(void) const { return actualCurrency_->asString(); }
friend std::ostream &operator<<(std::ostream &stream, Currency &currency)
	{
		stream << currency.asString();
		return stream;
	}
	bool operator<(const Currency &other) { return amountInEuro() < other.amountInEuro(); }
	bool operator==(const Currency &other) { return amountInEuro() == other.amountInEuro(); }
	bool operator>(const Currency &other) { return amountInEuro() > other.amountInEuro(); }
	bool operator<=(const Currency &other) { return amountInEuro() <= other.amountInEuro(); }
	bool operator>=(const Currency &other) { return amountInEuro() >= other.amountInEuro(); }
	Currency(CurrencyBase *derivedClassThis): actualCurrency_(derivedClassThis) { }
private:
	CurrencyBase *actualCurrency_;
};


class Euro: public CurrencyBase {
friend class Currency;
public:
	operator Currency(void) {
		return Currency(this->copy());
	}
	CurrencyBase *copy(void) const {
		return new Euro(amount_);
	}
	virtual Euro &operator+=(const Currency& amount) {
		amount_ += amount.amountInEuro();
		return *this;
	}
	virtual Euro &operator-=(const Currency& amount) {
		amount_ -= amount.amountInEuro();
		return *this;
	}
	virtual Euro &operator=(const Currency& amount) {
		amount_ = amount.amountInEuro();
		return *this;
	}
	Euro(double amount = 0.0): amount_(amount) { }
	virtual std::string name(void) const { return "Euro"; }
	virtual std::string sign(void) const { return "€"; }
	virtual double amountInEuro(void) const { return amount_; }
	virtual std::string asString(void) const {
		std::ostringstream stream;
		stream << amount_;
		return stream.str();
	}
private:
	double amount_;
};

inline Currency::Currency(void): actualCurrency_(new Euro) {  }


#endif
